# Define the file names
$originalFile = "physicstasksToCompare.ymt"
$newVersionFile = "new version.ymt"
$activeFile = "physicstasksWIP.ymt"
$backupFile = "physicstasksWIPBackup.ymt"  # Backup-Name gendert

# Check if 'physicstasksWIP.ymt' exists before running the script
if (-not (Test-Path $activeFile)) {
    Write-Host "Error: The file 'physicstasksWIP.ymt' was not found!" -ForegroundColor Red
    Read-Host "Press any key to exit the script..."
    exit
}

function Option1 {
    Clear-Host
    if (-not (Test-Path $originalFile)) {
        Write-Host "Couldn't find 'physicstasksToCompare.ymt'." -ForegroundColor Red
        return  # End the function if the file is not found
    }

    if (Test-Path $backupFile) {
        Write-Host "A backup of 'physicstasksWIP.ymt' already exists as 'physicstasksWIPBackup.ymt'. Please delete or restore it before creating a new one." -ForegroundColor Red
    } else {
        if (Test-Path $activeFile) {
            Rename-Item $activeFile $backupFile -Force
        }
        Copy-Item $originalFile $activeFile
        Write-Host "physicstasksWIPBackup.ymt has been created, and physicstasksToCompare.ymt has been copied as physicstasksWIP.ymt." -ForegroundColor Green
    }
}

function Option2 {
    Clear-Host
    if (Test-Path $backupFile) {
        if (Test-Path $activeFile) {
            Remove-Item $activeFile -Force
        }
        Rename-Item $backupFile $activeFile -Force
        Write-Host "Backup restored to physicstasksWIP.ymt." -ForegroundColor Green
    } else {
        Write-Host "No 'physicstasksWIPBackup.ymt' file found." -ForegroundColor Red
    }
}

function Option3 {
    Clear-Host
    $existingFiles = Get-ChildItem -Name "*new version*.ymt"

    if ($existingFiles.Count -eq 0) {
        $newIndex = 1
    } else {
        $highestFile = $existingFiles | ForEach-Object { [int]($_ -replace " .*") } | Sort-Object -Descending | Select-Object -First 1
        $newIndex = $highestFile + 1
    }

    $comment = Read-Host "Comment on this version"
    $newRifleFile = "$newIndex new version($comment).ymt"
    Copy-Item $activeFile $newRifleFile
    Write-Host "'physicstasksWIP.ymt' saved as $newRifleFile" -ForegroundColor Green
}

function Option4 {
    Clear-Host
    if (Test-Path $backupFile) {
        Write-Host "A backup of 'physicstasksWIP.ymt' already exists as 'physicstasksWIPBackup.ymt'. Please delete or restore it before creating a new one." -ForegroundColor Red
        return
    }

    $selection = Get-ChildItem -Name "*new version*.ymt"
    if ($selection.Count -eq 0) {
        Write-Host "No 'new version'-file found." -ForegroundColor Red
        return
    }

    $selection = @("Back") + $selection

    while ($true) {
        Clear-Host
        for ($i = 0; $i -lt $selection.Count; $i++) {
            $fileName = $selection[$i]
            Write-Host ("{0}: {1}" -f $i, $fileName)
        }

        $choice = Read-Host "Choose a file (enter the number)"
        
        # Check if input is a valid number and within range
        if ($choice -match '^\d+$' -and [int]$choice -ge 0 -and [int]$choice -lt $selection.Count) {
            if ($choice -eq 0) {
                ShowMenu
                return
            }

            $selectedFile = $selection[$choice]
            
            # Backup erstellen, wenn eine Datei ausgewhlt wurde
            if (-not (Test-Path $backupFile)) {
                if (Test-Path $activeFile) {
                    Copy-Item $activeFile $backupFile
                    Write-Host "'physicstasksWIPBackup.ymt' created." -ForegroundColor Yellow
                }
            }

            Copy-Item $selectedFile $activeFile -Force
            Write-Host "File '$selectedFile' copied to physicstasksWIP.ymt" -ForegroundColor Green
            break
        } else {
            # Error message for invalid input
            Write-Host "Invalid input!" -ForegroundColor Red
            Read-Host "Press any key to try again..."
        }
    }
}

function Option9 {
    Clear-Host
    if (Test-Path $backupFile) {
        Remove-Item $backupFile -Force
        Write-Host "'physicstasksWIPBackup.ymt' deleted!" -ForegroundColor Green
    } else {
        Write-Host "Couldn't find 'physicstasksWIPBackup.ymt'" -ForegroundColor Red
    }
}

function ShowMenu {
    Clear-Host
    Write-Host "1: Copy 'physicstasksToCompare.ymt' as 'physicstasksWIP.ymt'. A Backup will be created first."
    Write-Host "2: Restore backup. 'physicstasksWIPBackup.ymt' will be copied over current 'physicstasksWIP.ymt'."
    Write-Host "3: Create a new version of the current 'physicstasksWIP.ymt'."
    Write-Host "4: List and choose a 'new version' to restore. A Backup will be created first."
    Write-Host "9: Delete backup. 'physicstasksWIPBackup.ymt' will be deleted."
    Write-Host "0: Exit."
}

$exit = $false

while (-not $exit) {
    ShowMenu
    $choice = Read-Host "Choose an option"
    switch ($choice) {
        1 { Option1 }
        2 { Option2 }
        3 { Option3 }
        4 { Option4 }
        9 { Option9 }
        0 { $exit = $true }
        default { Write-Host "Error. Please choose an existing option!" -ForegroundColor Red }
    }
    Pause
}

Clear-Host
Write-Host "Script terminated."
